import React, { useState, useEffect, useRef } from 'react';
import { X, Save, Copy, Maximize2, Minimize2 } from 'lucide-react';

export function CellPopover({ isOpen, anchorRect, value, onSave, onClose, title }) {
    const [content, setContent] = useState(value);
    const [isExpanded, setIsExpanded] = useState(false);

    // Update content when value changes
    useEffect(() => {
        setContent(value);
    }, [value]);

    if (!isOpen) return null;

    // Responsive positioning logic
    const getStyle = () => {
        if (isExpanded) {
            return {
                position: 'fixed',
                top: '50%',
                left: '50%',
                transform: 'translate(-50%, -50%)',
                width: '90vw',
                height: '85vh',
                maxWidth: '1000px',
                zIndex: 1000
            };
        }

        // Default small popover
        if (!anchorRect) return { display: 'none' };

        const top = anchorRect.bottom + 5;
        const left = anchorRect.left;

        // Adjust if off-screen
        const safeLeft = Math.min(left, window.innerWidth - 420); // Keep 420px width on screen
        const safeTop = Math.min(top, window.innerHeight - 320); // Keep 300px height on screen

        return {
            position: 'fixed',
            top: safeTop,
            left: safeLeft,
            width: '400px',
            height: '300px',
            zIndex: 100
        };
    };

    const style = getStyle();

    return (
        <>
            {/* Backdrop */}
            <div
                className="fixed inset-0 z-[90]"
                onClick={onClose}
                style={{
                    background: isExpanded ? 'rgba(0,0,0,0.6)' : 'transparent',
                    pointerEvents: 'auto'
                }}
            />

            <div
                className={`flex flex-col bg-[var(--bg-secondary)] border border-[var(--border-color)] rounded-lg shadow-2xl overflow-hidden transition-all duration-200 ease-in-out ${isExpanded ? 'scale-100 opacity-100' : 'scale-95 opacity-100'}`}
                style={style}
            >
                {/* Header */}
                <div className="flex items-center justify-between p-3 border-b border-[var(--border-color)] bg-[var(--bg-tertiary)]">
                    <span className="font-semibold text-sm text-[var(--accent-primary)] uppercase tracking-wider">
                        {title || 'Cell Editor'}
                    </span>
                    <div className="flex items-center gap-1">
                        <button
                            onClick={() => setIsExpanded(!isExpanded)}
                            className="p-1.5 rounded hover:bg-[var(--bg-primary)] text-[var(--text-muted)] transition-colors"
                            title={isExpanded ? "Collapse" : "Expand"}
                        >
                            {isExpanded ? <Minimize2 size={16} /> : <Maximize2 size={16} />}
                        </button>
                        <button
                            onClick={() => navigator.clipboard.writeText(content)}
                            className="p-1.5 rounded hover:bg-[var(--bg-primary)] text-[var(--text-muted)] transition-colors"
                            title="Copy to Clipboard"
                        >
                            <Copy size={16} />
                        </button>
                        <button
                            onClick={onClose}
                            className="p-1.5 rounded hover:bg-[var(--bg-primary)] text-[var(--text-muted)] hover:text-red-500 transition-colors"
                            title="Close"
                        >
                            <X size={18} />
                        </button>
                    </div>
                </div>

                {/* Editor Area */}
                <div className="flex-1 relative">
                    <textarea
                        value={content || ''}
                        onChange={(e) => setContent(e.target.value)}
                        className="w-full h-full p-4 bg-[var(--bg-primary)] text-[var(--text-primary)] font-mono text-sm resize-none focus:outline-none leading-relaxed"
                        placeholder="Type here..."
                        spellCheck={false}
                        autoFocus
                    />
                </div>

                {/* Footer */}
                <div className="flex items-center justify-between p-3 border-t border-[var(--border-color)] bg-[var(--bg-tertiary)]">
                    <span className="text-xs text-[var(--text-muted)]">
                        {content ? content.length : 0} characters
                    </span>
                    <div className="flex gap-2">
                        <button
                            onClick={onClose}
                            className="px-3 py-1.5 rounded text-xs font-medium text-[var(--text-primary)] hover:bg-[var(--bg-primary)] transition-colors"
                        >
                            Cancel
                        </button>
                        <button
                            onClick={() => { onSave(content); onClose(); }}
                            className="px-4 py-1.5 rounded text-xs font-medium text-white bg-[var(--accent-primary)] hover:opacity-90 shadow-sm transition-all"
                        >
                            Save Changes
                        </button>
                    </div>
                </div>
            </div>
        </>
    );
}
